<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
    public function __construct()
    {
        $this->middleware('guest')->except('logout');
    }

    public function register()
    {
        return view('auth/register');
    }

    public function registerSave(Request $request)
    {
        // Custom password validation rules
        $rules = [
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users',
            'phoneNum' => 'required|regex:/^[0-9]{10,15}$/',
            'password' => [
                'required',
                'confirmed',
                'regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)[A-Za-z\d]{6,}$/',
            ],
        ];

        $messages = [
            'password.regex' => 'Kata laluan mesti memenuhi kriteria berikut:<ul>
                <li>Mengandungi sekurang-kurangnya 8 aksara.</li>
                <li>Termasuk sekurang-kurangnya satu huruf besar (A-Z).</li>
                <li>Termasuk sekurang-kurangnya satu huruf kecil (a-z).</li>
                <li>Termasuk sekurang-kurangnya satu nombor (0-9).</li>
            </ul>',
            'password.confirmed' => 'Kata laluan dan pengesahan kata laluan tidak sepadan.',
        ];

        Validator::make($request->all(), $rules, $messages)->validate();

        User::create([
            'name' => $request->name,
            'email' => $request->email,
            'phoneNum' => $request->phoneNum,
            'password' => Hash::make($request->password),
            'role' => "0",
        ]);

        return redirect()->route('login')->with('success', 'Pendaftaran berjaya! Sila log masuk.');
    }


    public function login()
    {
        return view('auth/login');
    }

    public function loginAction(Request $request)
    {
        Validator::make($request->all(), [
            'email' => 'required|email',
            'password' => 'required',
        ])->validate();

        if (!Auth::attempt($request->only('email', 'password'), $request->boolean('remember'))) {
            return back()->with('error', 'Maklumat log masuk tidak sah. Sila cuba lagi.');
        }

        $request->session()->regenerate();

        if (auth()->user()->role == 'admin') {
            return redirect()->route('admin.homepage');
        } else if (auth()->user()->role == 'owner') {
            return redirect()->route('owner.homepage');
        } else {
            return redirect()->route('user.homepage');
        }
    }

    public function logout(Request $request)
    {
        Auth::guard('web')->logout(); // Log the user out

        $request->session()->invalidate(); // Invalidate session
        $request->session()->regenerateToken(); // Regenerate CSRF token

        return redirect('/home'); // Redirect to homepage after logout
    }
}
