<?php

namespace App\Http\Controllers;

use App\Models\Booking;
use App\Models\Invoice;
use App\Models\Payment;
use App\Models\Quotation;
use Illuminate\Http\Request;

class InvoiceController extends Controller
{


    public function index()
    {
        // Check if the user is authenticated
        if (!auth()->check()) {
            // Redirect to login or show an error
            return redirect()->route('login')->with('error', 'Sila log masuk untuk mengakses halaman ini.');
        }

        $role = auth()->user()->role;

        if (!in_array($role, ['admin', 'owner'])) {
            abort(403, 'Unauthorized action.');
        }

        $layout = $role === 'admin' ? 'layouts.baseAdmin' : 'layouts.baseSP';

        $bookings = Booking::whereIn('status', ['Kerja Dalam Proses', 'Selesai'])
            ->with(['user', 'service', 'subService', 'quotation', 'payments'])
            ->get();

        return view('Admin.invoice.index', compact('layout', 'bookings'));
    }

    public function create(Request $request, $id)
    {
        try {
            // Validate booking
            $booking = Booking::findOrFail($id);

            // Validate payment
            $payment = Payment::findOrFail($request->payment_id);
            if ($payment->booking_id != $id) {
                throw new \Exception('ID pembayaran tidak sepadan dengan tempahan ini.');
            }

            // Check booking status
            if ($booking->status != 'Kerja Dalam Proses') {
                throw new \Exception('Tempahan bukan kerja dalam proses.');
            }

            // Update booking status
            $booking->status = 'Selesai';
            $booking->save();

            // Create invoice
            Invoice::create([
                'booking_id' => $booking->id,
                'payment_id' => $payment->id,
                'amount' => $booking->total_payment,
            ]);

            // Redirect based on user role
            $route = auth()->user()->role === 'admin' ? 'invoice.index' : 'invoice.list';
            return redirect()->route($route)->with('success', 'Status berjaya dikemas kini dan invois dicipta.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Terdapat ralat semasa mengemas kini status atau mencipta invois. ' . $e->getMessage());
        }
    }



    //Customer
    public function show(Request $request, $id)
    {
        // Fetch the booking using the ID, with necessary relationships
        $booking = Booking::whereIn('status', ['Kerja Dalam Proses', 'Selesai'])
            ->with(['user', 'address', 'quotation'])
            ->findOrFail($id); // Ensure the booking exists

        // Fetch the invoice associated with this booking
        $invoice = Invoice::where('booking_id', $booking->id)->with('payment')->first();

        if (!$invoice) {
            // Handle the case where no invoice is found for the booking
            return redirect()->back()->with('error', 'Tiada invois tersedia untuk tempahan ini.');
        }

        // Fetch all quotation items related to this booking
        $items = Quotation::where('booking_id', $booking->id)->get();

        // Calculate the total amount dynamically by summing the total_price for all items
        $totalAmount = $items->sum('total_price');

        // Pass the data to the view
        return view('invoice.show', compact('booking', 'invoice', 'items', 'totalAmount'));
    }


    public function display(Request $request, $id)
    {
        // Check if the user is authenticated
        if (!auth()->check()) {
            // Redirect to login or show an error
            return redirect()->route('login')->with('error', 'Sila log masuk untuk mengakses halaman ini.');
        }

        $role = auth()->user()->role;

        if (!in_array($role, ['admin', 'owner'])) {
            abort(403, 'Unauthorized action.');
        }

        $layout = $role === 'admin' ? 'layouts.baseAdmin' : 'layouts.baseSP';

        // Fetch the booking using the ID, with necessary relationships
        $booking = Booking::whereIn('status', ['Kerja Dalam Proses', 'Selesai'])
            ->with(['user', 'address', 'quotation'])
            ->findOrFail($id); // Ensure the booking exists

        // Fetch the invoice associated with this booking
        $invoice = Invoice::where('booking_id', $booking->id)->with('payment')->first();

        if (!$invoice) {
            // Handle the case where no invoice is found for the booking
            return redirect()->back()->with('error', 'Tiada invois tersedia untuk tempahan ini.');
        }

        // Fetch all quotation items related to this booking
        $items = Quotation::where('booking_id', $booking->id)->get();

        // Calculate the total amount dynamically by summing the total_price for all items
        $totalAmount = $items->sum('total_price');

        // Pass the data to the view
        return view('Admin.invoice.show', compact('layout', 'booking', 'invoice', 'items', 'totalAmount'));
    }

    public function view()
    {
        return view('Admin.invoice.show'); //admin view invoice
    }
    
}
