<?php

namespace App\Http\Controllers;

use App\Models\Booking;
use App\Models\Quotation;
use App\Models\SubService;
use Illuminate\Http\Request;

class QuotationController extends Controller
{

//Customer

    public function show($id)
    {
        // Fetch the quotation using booking_id
        $quotation = Quotation::where('booking_id', $id)
            ->with(['booking', 'booking.user', 'booking.address'])
            ->firstOrFail(); // Ensure the quotation exists

        // Fetch all quotation items related to this booking
        $items = Quotation::where('booking_id', $quotation->booking_id)->get();

        // Calculate the total amount dynamically by summing the total_price for all quotation items
        $totalAmount = $items->sum('total_price');

        // Pass the data to the view
        return view('quotation.show', compact('quotation', 'items', 'totalAmount'));
    }

    public function reject(Request $request, $id)
    {
        $quotation = Quotation::findOrFail($id);
        $booking = $quotation->booking;

        // Validate the request
        $request->validate([
            'reason' => 'required|string',
            'otherReason' => 'nullable|string|max:255',
            'newQuotation' => 'nullable|boolean',
        ]);

        // Check if the selected reason is "Lain-Lain" and handle the custom reason
        $rejectionReason = $request->reason;
        if ($rejectionReason === 'Lain-Lain') {
            $rejectionReason = $request->otherReason ?: 'Tidak dinyatakan'; // Use the custom reason or default text
        }

        // Update booking status and rejection reason
        $booking->status = $request->has('newQuotation') && $request->newQuotation ? 'Menunggu Sebut Harga' : 'Tolak';
        $booking->rejection_reason = $rejectionReason;
        $booking->save();

        // If the status is 'Menunggu Sebut Harga', notify the owner to edit the quotation (optional feature)
        // if ($booking->status === 'Menunggu Sebut Harga') {
        //     // Optionally, trigger a notification to the owner
        //     // Notification::send($owner, new RequestNewQuotationNotification($booking));
        // }

        return redirect()
            ->route('booking.progress', ['id' => $booking->id])
            ->with('success', 'Sebut Harga berjaya ditolak.');
    }


    public function accept($id)
    {
        $quotation = Quotation::findOrFail($id);
        $booking = $quotation->booking;

        // Update the status to 'Belum Bayar'
        $booking->status = 'Belum Bayar';
        $booking->save();

        // Redirect to the booking progress page with the required booking ID
        return redirect()->route('booking.progress', ['id' => $booking->id])->with('success', 'Sebut Harga telah diterima.');
    }




//Admin

    //Create Quotation
    public function create($bookingId)
    {
        // Retrieve the booking with the related data (user, address, service, subservice)
        $booking = Booking::with(['user', 'address', 'service', 'subService'])->findOrFail($bookingId);

        // Retrieve the first subService associated with the booking (assuming 1-to-1 relationship)
        $subService = $booking->subService; // Assuming subService is related directly to the booking

        return view('Admin.quotation.create', compact('booking', 'subService'));
    }

    //Store Quotation Details 
    public function store(Request $request, $bookingId)
    {
        // Validate incoming data
        $validated = $request->validate([
            'scope_of_work' => 'array|required',
            'scope_of_work.*' => 'nullable|string',
            'quantity' => 'array|required',
            'quantity.*' => 'numeric|min:1',
            'unit_price' => 'array|required',
            'unit_price.*' => 'numeric|min:0',
            'total_price' => 'array|required',
            'total_price.*' => 'numeric|min:0',
        ]);

        // Retrieve the booking
        $booking = Booking::findOrFail($bookingId);

        
        $totalPayment = 0;  // Initialize the total payment

        // Create the quotation entries
        foreach ($request->scope_of_work as $index => $scope) {
            // Calculate total price for each row
            $totalPrice = $request->quantity[$index] * $request->unit_price[$index];

            // Create the quotation entry
            Quotation::create([
                'booking_id' => $bookingId,
                'scope_of_work' => $scope,
                'quantity' => $request->quantity[$index],
                'unit_price' => $request->unit_price[$index],
                'total_price' => $totalPrice,
            ]);

            // Add to the total payment
            $totalPayment += $totalPrice;
        }


        // After saving all quotations, update the booking's total payment
        $booking->update(['status' => 'Sebut Harga', 'total_payment' => $totalPayment]);

        // Redirect to the quotation list page with a success message
        return redirect()->route('quotation.list')->with('success', 'Sebut Harga telah berjaya disimpan.');
    }


    // View Quotation
    public function view($id)
    {
        // Fetch the quotation with related booking, user, and address
        $quotation = Quotation::with(['booking', 'booking.user', 'booking.address'])->findOrFail($id);

        // Fetch all quotation items related to this booking
        $items = Quotation::where('booking_id', $quotation->booking_id)->get();

        // Calculate the total amount
        $totalAmount = $items->sum('total_price');

        // Determine the view based on the user's role
        $view = auth()->user()->role === 'admin' ? 'Admin.quotation.show' : 'Owner.quotation.show';

        // Return the appropriate view
        return view($view, compact('quotation', 'items', 'totalAmount'));
    }


    public function index()
    {
        // Get the user's role
        $role = auth()->user()->role;

        if ($role === 'admin') { // Admin
            $bookings = Booking::whereIn('status', ['Lulus', 'Tolak', 'Sebut Harga', 'Menunggu Sebut Harga'])
                ->with(['user', 'service', 'subService', 'quotation'])
                ->get();

            foreach ($bookings as $booking) {
                $booking->quotation_exists = $booking->quotation()->exists();
            }

            $view = 'Admin.quotation.index';

        } elseif ($role === 'owner') { // Owner
            $bookings = Booking::whereHas('quotation')
                ->with(['user', 'service', 'quotation'])
                ->get();

            $view = 'Owner.quotation.index';
        } else {
            abort(403, 'Unauthorized action.');
        }

        return view($view, compact('bookings'));
    }




//Owner

    // Edit Quotation
    public function edit($id)
    {
        // Fetch the quotation with related booking
        $quotation = Quotation::with(['booking', 'booking.user', 'booking.address', 'booking.service', 'booking.subService'])->findOrFail($id);

        // Retrieve all quotation items related to the booking
        $items = Quotation::where('booking_id', $quotation->booking_id)->get();

        // Calculate total amount
        $totalAmount = $items->sum('total_price');

        // Determine user role
        $userRole = auth()->user()->role; // Assume role is stored in the user model

        if ($userRole === 'admin') {
            return view('Admin.quotation.edit', compact('quotation', 'items', 'totalAmount'));
        } elseif ($userRole === 'owner') {
            return view('Owner.quotation.edit', compact('quotation', 'items', 'totalAmount'));
        } else {
            abort(403, 'Unauthorized action.');
        }
    }

    public function update(Request $request, $bookingId)
    {
        // Validate incoming data
        $validated = $request->validate([
            'scope_of_work.*' => 'required|string',
            'quantity.*' => 'required|integer|min:1',
            'unit_price.*' => 'required|numeric|min:0',
            'total_price.*' => 'required|numeric|min:0',
        ]);

        // Retrieve the booking and quotation
        $booking = Booking::findOrFail($bookingId);

        // Determine user role
        $userRole = auth()->user()->role; // Assume role is stored in the user model

        if ($userRole === 'admin' || $userRole === 'owner') {

            // Owner-specific logic: Change status if 'Menunggu Sebut Harga'
            if ($booking->status === 'Menunggu Sebut Harga') {
                $booking->status = 'Sebut Harga Baru';
                // $booking->save();
            }

            // Handle deletion of rows
            if ($request->has('deleted_rows')) {
                $deletedRows = json_decode($request->deleted_rows, true); // Parse the JSON string
                if (!empty($deletedRows)) {
                    Quotation::whereIn('id', $deletedRows)->delete(); // Delete rows from the database
                }
            }

            $totalPayment = 0;

            // Retrieve existing quotations for the booking
            $existingItems = Quotation::where('booking_id', $bookingId)->get();

            foreach ($request->scope_of_work as $index => $scope) {

                $totalPrice = $request->quantity[$index] * $request->unit_price[$index];
                $totalPayment += $totalPrice;

                if (isset($existingItems[$index])) {
                    // Update existing item
                    $existingItems[$index]->update([
                        'scope_of_work' => $scope,
                        'quantity' => $request->quantity[$index],
                        'unit_price' => $request->unit_price[$index],
                        'total_price' => $totalPrice,
                    ]);
                } else {
                    // Create new item
                    Quotation::create([
                        'booking_id' => $bookingId,
                        'scope_of_work' => $scope,
                        'quantity' => $request->quantity[$index],
                        'unit_price' => $request->unit_price[$index],
                        'total_price' => $totalPrice,
                    ]);
                }
            }
            

            // Update booking's total payment
            $booking->update(['total_payment' => $totalPayment]);

            // Redirect based on user role
            if ($userRole === 'admin') {
                return redirect()->route('quotation.list')->with('success', 'Sebut Harga berjaya dikemaskini.');
            } else {
                return redirect()->route('quotation.index')->with('success', 'Sebut Harga berjaya dikemaskini.');
            }
        }

        // If user is not admin or owner
        abort(403, 'Unauthorized action.');
    }



}
