<?php

namespace App\Http\Controllers;

use App\Models\Address;
use App\Models\Booking;
use App\Models\Service;
use App\Models\SubService;
use Illuminate\Http\Request;

class ServiceController extends Controller
{

    //Admin
    public function index()
    {
        $services = Service::with('subServices')->get(); // Eager load sub-services

        return view('Admin.service.index', compact('services'));
    }


    public function create()
    {
        return view('Admin.service.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'service_name' => 'required|string|max:255',
            'sub_services' => 'required|array',
            'sub_services.*.name' => 'required|string|max:255',
            'sub_services.*.price' => 'required|numeric|min:0',
            'sub_services.*.estimate_price' => 'required|numeric|min:0',
        ]);

        // Create the main service
        $service = Service::create([
            'name' => $request->service_name,
        ]);

        // Add sub-services
        foreach ($request->sub_services as $subService) {
            SubService::create([
                'service_id' => $service->id,
                'name' => $subService['name'],
                'price' => $subService['price'],
                'estimate_price' => $subService['estimate_price'],
            ]);
        }

        return redirect()->route('service.index')->with('success', 'Servis berjaya dicipta!');
    }


    public function edit($id)
    {
        $service = Service::with('subServices')->findOrFail($id); // Retrieve a single Service
        $booking = Booking::with('address')->findOrFail($id); // If related bookings are required
        return view('Admin.service.edit', compact('service', 'booking'));
    }

    public function update(Request $request, Service $service)
    {
        // Validate the incoming request
        $request->validate([
            'service_name' => 'required|string|max:255',
            'sub_services' => 'required|array',
            'sub_services.*.name' => 'required|string|max:255',
            'sub_services.*.price' => 'required|numeric|min:0',
            'sub_services.*.estimate_price' => 'required|numeric|min:0',
        ]);

        // Update the main service
        $service->update([
            'name' => $request->service_name,
        ]);

        // Remove existing sub-services and add updated ones
        $service->subServices()->delete(); // Optional: Remove existing sub-services

        foreach ($request->sub_services as $subService) {
            SubService::create([
                'service_id' => $service->id,
                'name' => $subService['name'],
                'price' => $subService['price'],
                'estimate_price' => $subService['estimate_price'],
            ]);
        }

        return redirect()->route('service.index')->with('success', 'Servis berjaya dikemaskini!');
    }



    public function show(Service $service)
    {
        return view('Admin.service.show', compact('service'));
    }

    public function destroy(Service $service)
    {
        $service->delete(); // Delete the service record
        return redirect()->route('service.index')->with('success', 'Servis berjaya dipadam!');
    }

    //Owner

    public function list()
    {
        $services = Service::with('subServices')->get(); // Eager load sub-services

        return view('Owner.service.index', compact('services'));
    }

    public function editList($id)
    {
        $service = Service::with('subServices')->findOrFail($id); // Retrieve a single Service
        $booking = Booking::with('address')->findOrFail($id); // If related bookings are required

        return view('Owner.service.edit', compact('service', 'booking'));
    }

    public function updateList(Request $request, Service $service)
    {
        // Validate the incoming request
        $request->validate([
            'service_name' => 'required|string|max:255',
            'sub_services' => 'required|array',
            'sub_services.*.name' => 'required|string|max:255',
            'sub_services.*.price' => 'required|numeric|min:0',
            'sub_services.*.estimate_price' => 'required|numeric|min:0',
        ]);

        // Update the main service
        $service->update([
            'name' => $request->service_name,
        ]);

        // Remove existing sub-services and add updated ones
        $service->subServices()->delete(); // Optional: Remove existing sub-services

        foreach ($request->sub_services as $subService) {
            SubService::create([
                'service_id' => $service->id,
                'name' => $subService['name'],
                'price' => $subService['price'],
                'estimate_price' => $subService['estimate_price'],
            ]);
        }

        return redirect()->route('service.list')->with('success', 'Servis berjaya dikemaskini!');
    }



    public function showList(Service $service)
    {
        return view('Owner.service.show', compact('service'));
    }

    public function delete(Service $service)
    {
        $service->delete(); // Delete the service record
        return redirect()->route('service.list')->with('success', 'Servis berjaya dipadam!');
    }
}
