<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\PlantDetail;
use Illuminate\Http\Request;

class PlantDetailController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $plantDetails = PlantDetail::with('category')->get();
        $categories = Category::all();

        // Calculate the count of plant_type for each category
        $plantTypeCountByCategory = [];
        foreach ($categories as $category) {
            $plantTypeCountByCategory[$category->id] = PlantDetail::where('category_id', $category->id)
                ->distinct('plant_type') // Ensure unique plant types
                ->count('plant_type');
        }

        // Calculate the total number of unique plant types across all categories
        $totalPlantTypes = PlantDetail::distinct('plant_type')->count('plant_type');

       return view('plant.index', compact('plantDetails', 'categories', 'plantTypeCountByCategory', 'totalPlantTypes'))->with('success', session('success'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $categories = Category::all();
        return view('plant.index', compact('categories'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \App\Http\Requests\StoreplantdetailsRequest  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'plant_category' => 'required|exists:categories,id', // Validate against categories table
            'plant_type' => 'required|string|max:255',
            'plant_maturity' => 'required|integer',
        ]);
        $plantDetails = new PlantDetail();
        $plantDetails->category_id = $request->plant_category;
        $plantDetails->plant_type = $request->plant_type;
        $plantDetails->plant_maturity = $request->plant_maturity;

        $plantDetails->save();

        return redirect()->route('plant.index')->with('success', 'Maklumat Pokok Berjaya Disimpan');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\PlantDetail  $plantdetails
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $plantDetails = PlantDetail::with('category')->findOrFail($id);
        return view('plan.index', compact('plantDetail'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\PlantDetail  $plantdetails
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $plantDetail = PlantDetail::with('category')->findOrFail($id); // Eager load category
        $categories = Category::all(); // Get all categories for the dropdown
        return view('plant.index', compact('plantDetail', 'categories')); // Assuming you have a separate edit view
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \App\Http\Requests\UpdatePlantDetailRequest  $request
     * @param  \App\Models\PlantDetail  $plantdetails
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        // Validate only the plant_maturity field
        $request->validate([
            'plant_maturity' => 'required|integer',
        ]);

        // Find the plant by ID
        $plantDetail = PlantDetail::findOrFail($id);

        // Update only the plant_maturity field
        $plantDetail->plant_maturity = $request->plant_maturity;

        // Save the changes
        $plantDetail->save();


        return redirect()->route('plant.index')->with('success', 'Maklumat Pokok Berjaya Dikemaskini.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\PlantDetail  $plantdetails
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $plantDetail = PlantDetail::findOrFail($id);
        $plantDetail->delete();

        return redirect()->route('plant.index')->with('success', 'Maklumat Pokok Berjaya Dipadam.');
    }
}
