<?php

namespace App\Http\Controllers;
use App\Models\Category;
use App\Models\Farm;
use App\Models\Plantation;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;

class FarmController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {

        $farms = Farm::with('category')->get();
        $categories = Category::all();

        //calculate
        $totalFarms = $farms->count();
        $totalCategories = $categories->count();
        $totalFarmsWithPlantations = Farm::whereHas('plantations', function ($query) {
            $query->whereIn('status', ['Baru', 'Tanam']);
        })->count();
    
      
        return view('farm.index', compact('farms', 'categories', 'totalFarms', 'totalCategories', 'totalFarmsWithPlantations'))->with('success', session('success'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $categories = Category::all(); 
        return view('farm.create', compact('categories'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \App\Http\Requests\StoreFarmRequest  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'plant_category' => 'required|exists:categories,id',
            'name' => 'required|string|max:255',
            'size' => 'required|numeric',
            'type' => 'required|string',
            'address' => 'required|string',
            'pdf' => 'nullable|file|mimes:pdf|max:2048',
        ]);

        $farm = new Farm();
        $farm->category_id = $request->plant_category;
        $farm->name = $request->name;
        $farm->size = $request->size;
        $farm->type = $request->type;
        $farm->address = $request->address;

        if ($request->hasFile('pdf')) {
            $originalFileName = $request->file('pdf')->getClientOriginalName(); // Get original file name
            $pdfPath = $request->file('pdf')->storeAs('public/banner', $originalFileName); // Store with original name
            $farm->pdf = $pdfPath; // Save the path to the database
        }

        $farm->save();

        return redirect()->route('farm.index')->with('success', 'Kebun Berjaya Disimpan.');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Farm  $farm
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $farm = Farm::find($id);
        return view('farm.show', compact('farm'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Farm  $farm
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $farm = Farm::findOrFail($id); 
        $categories = Category::all(); 
      
        return view('farm.edit', compact('farm', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \App\Http\Requests\UpdateFarmRequest  $request
     * @param  \App\Models\Farm  $farm
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
       
        $request->validate([
            'plant_category' => 'required|exists:categories,id',
            'name' => 'required|string|max:255',
            'size' => 'required|numeric',
            'type' => 'required|string',
            'address' => 'required|string',
            'pdf' => 'nullable|file|mimes:pdf|max:2048', // Optional PDF upload
        ]);

        // Find the farm by ID
        $farm = Farm::findOrFail($id);

        // Update the farm's attributes
        $farm->category_id = $request->plant_category;
        $farm->name = $request->name;
        $farm->size = $request->size;
        $farm->type = $request->type;
        $farm->address = $request->address;

        // Handle PDF upload if a new file is provided
        if ($request->hasFile('pdf')) {
            // Delete the old PDF file if it exists
            if ($farm->pdf) {
                Storage::delete($farm->pdf);
            }

            // Store the new PDF file and update the path
            $originalFileName = $request->file('pdf')->getClientOriginalName();
            $pdfPath = $request->file('pdf')->storeAs('public/banner', $originalFileName);
            $farm->pdf = $pdfPath; // Save the new path to the database
        }

        // Save the updated farm details
        $farm->save();

        // Redirect back to the index page with a success message
        return redirect()->route('farm.index')->with('success', 'Kebun Berjaya Dikemaskini.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Farm  $farm
     * @return \Illuminate\Http\Response
     */
    public function destroy(Farm $farm)
    {
        // Delete the farm record
        $farm->delete();

        // Redirect back with a success message
        return redirect()->route('farm.index')->with('success', 'Maklumat kebun telah dipadam.');
    }
}
